<?php

namespace OneSender\Salesloo;

defined('ABSPATH') or die('No direct access allowed to this file');

class Utils {

    public static function phone($phone) {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        if (substr($phone, 0, 2) === '08') {
            $phone = '628' . substr($phone, 2);
        }
        return $phone;
    }

    public static function rupiah($amount) {
        $amount = salesloo_convert_price($amount);
        return 'Rp ' . number_format($amount, 0, ',', '.');
    }

    public static function get_base_api_url( $url ): string {
	    $parsed    = parse_url( $url );
	    $protocol  = isset( $parsed['scheme'] ) ? $parsed['scheme'] : '';
	    $host      = isset( $parsed['host'] ) ? $parsed['host'] : '';
	    $port      = isset( $parsed['port'] ) ? ':' . $parsed['port'] : '';
	    $base_url  = $protocol . '://' . $host . $port;
	    return $base_url;
	}

    public static function get_api_url(): string {
        $url = salesloo_get_option('onesender_apiurl');
        $base_url = self::get_base_api_url($url);
        return $base_url . '/api/v1/messages';
    }

    public static function send_messages($messages = array()) {
        if (empty($messages)) return false;
    
        $remote_url = self::get_api_url();
        $token = salesloo_get_option('onesender_apikey');
    
        $headers = array(
            'Authorization: Bearer ' . $token,
            'Content-Type: application/json'
        );
    
        $curl = curl_init();
    
        curl_setopt_array(
            $curl,
            array(
                CURLOPT_URL => $remote_url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 3,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => json_encode($messages),
                CURLOPT_HTTPHEADER => $headers,
                CURLOPT_SSL_VERIFYPEER => false, 
                CURLOPT_SSL_VERIFYHOST => false, 
            )
        );
    
        $response = curl_exec($curl);
        $curlError = curl_error($curl);
    
        curl_close($curl);

        $array = json_decode($response);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [array(), $curlError];
        }
    
        return array($array, $curlError);
    }
    
    public static function parse_shortcodes( string $text, array $data ): string {
        $pattern = '/\{ ?([^}]+) ?\}/s';
        preg_match_all($pattern, $text, $matches);
    
        foreach ($matches[1] as $match) {
            $key = trim($match);
            $replacement = isset($data[$key]) ? $data[$key] : ''; 
            $text = str_replace("{{$match}}", $replacement, $text);
        }
    
        return $text;
    }

    public static function make_text($phone, $message) {
        $message = htmlspecialchars_decode($message);
        return [
            'to' => $phone,
            'recipient_type' => strpos($phone, '@g.us') === false ? 'individual' : 'group',
            'type' => 'text',
            'text' => [
                'body' => $message,
            ],
        ]; 
    }

    public static function admin_phones(): array {
        $phone_string = trim(salesloo_get_option('onesender_admins_phone'));
        if (empty($phone_string)) {
            return [];
        }

        $phones = explode(',', $phone_string);
        $output = array_map([self::class, 'phone'], $phones);

        return $output;
    }

    public static function cors() {
        if (isset($_SERVER['HTTP_ORIGIN'])) {
            header("Access-Control-Allow-Origin: {$_SERVER['HTTP_ORIGIN']}");
            header('Access-Control-Allow-Credentials: true');
        }
        
        if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
            
            if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_METHOD']))
                header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
            
            if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']))
                header("Access-Control-Allow-Headers: {$_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']}");
        
            exit(0);
        }
    }

    public static function auto_format_api_url($url) {
        if( filter_var( $url, FILTER_VALIDATE_URL ) === false )
            return;

        $url_parts = parse_url($url);

        $url_parts = wp_parse_args( $url_parts, array(
            'scheme'    => $url_parts['scheme'],
            'host'      => $url_parts['host'],
            'port'      => false,
            'path'      => $url_parts['path']
        ) );
        
        $api_server = sprintf( '%s://%s/api/v1/messages',
            $url_parts['scheme'],
            $url_parts['host']
        );

        if ( !empty( $url_parts['port'] ) ) {
            $api_server = sprintf( '%s://%s:%s/api/v1/messages',
                $url_parts['scheme'],
                $url_parts['host'],
                $url_parts['port']
            );
        }

        $api_server = apply_filters( 'onesender/salesloo/api_url', $api_server );
        return $api_server;
    }

    public static function request_json() {
        $raw    = file_get_contents( 'php://input' );
        $json   = json_decode( $raw, true );
    
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            return array();
        }
    
        foreach($json as $key => $value) {
            if ( is_string( $value ) ) {
                $json[$key] = sanitize_text_field( $value );
            } 
        }
        return $json;
    }

    public static function get_affiliate_data_from_invoice( $invoice_id ): array {

        $invoice_id = intval($invoice_id);

        global $wpdb;

        $output = [
            'affiliate_name'  => '',
            'affiliate_email' => '',
            'affiliate_phone' => '',
            'commission'      => '',
        ];

        $table_name = $wpdb->prefix . 'salesloo_commission';

        $query = $wpdb->prepare(
            "SELECT user_id, amount FROM $table_name WHERE invoice_id = %d",
            $invoice_id
        );

        $result = $wpdb->get_row($query, ARRAY_A);

        if($result) {
            $affiliate = get_user_by('id', $result['user_id']);
            $output['affiliate_name'] = $affiliate->data->display_name ?? '';
            $output['affiliate_email'] = $affiliate->data->user_email ?? '';
            $output['commission'] = Utils::rupiah( $result['amount'] ?? '');

            $phone = Utils::phone( get_user_meta( intval($result['user_id'] ?? 0), 'phone', true ) );
            $output['affiliate_phone'] = strval($phone);
        }

        return $output;
    }
    
    public static function generate_box_key() {
        return base64_encode( openssl_random_pseudo_bytes( 32 ) );
    }
}
